<# 
WORG installer script for Windows 10/11
- Sets/uses user-scoped env vars (WORG_HOME, WORG_PYTHON)
- Creates venv under WORG_HOME and installs requirements
- Avoids requiring admin when possible
#>

Set-ExecutionPolicy -Scope Process -ExecutionPolicy Bypass -Force

function Show-Step($msg) {
    Write-Host ""
    Write-Host "=== $msg ===" -ForegroundColor Cyan
}

function Pause-For-User($msg) {
    Write-Host $msg -ForegroundColor Yellow
    Read-Host "Press ENTER to continue"
}

function Recommend-WorgHome {
    return (Join-Path $Env:USERPROFILE "Documents\WORG")
}

# ---------------- step 0: choose/validate WORG_HOME ----------------

Show-Step "Step 0 - Choosing / validating WORG_HOME"

$worgHome = $Env:WORG_HOME

if (-not $worgHome -or $worgHome.Trim() -eq "") {
    Write-Host "WORG_HOME is not set." -ForegroundColor Yellow
    Write-Host ("Recommended location (no admin needed): {0}" -f (Recommend-WorgHome)) -ForegroundColor Green
    $worgHome = Read-Host "Enter full path for WORG_HOME (or press Enter to cancel)"
    if (-not $worgHome -or $worgHome.Trim() -eq "") {
        Write-Host "No location selected. Re-run and pick a folder under your user profile (Documents recommended)." -ForegroundColor Red
        exit 1
    }
    $Env:WORG_HOME = $worgHome
    setx WORG_HOME "$worgHome" > $null
}

# Create folder if missing (user-writable locations should succeed without admin)
if (-not (Test-Path -LiteralPath $worgHome)) {
    New-Item -ItemType Directory -Path $worgHome -Force | Out-Null
}

# Validate code presence
$rfwPath = Join-Path $worgHome "worg_proc.py"
if (-not (Test-Path -LiteralPath $rfwPath)) {
    Write-Host ("WORG_HOME is '{0}', but worg_proc.py was not found there." -f $worgHome) -ForegroundColor Red
    Write-Host "Place the WORG code in that folder (or choose the folder that already contains worg_proc.py) and re-run."
    exit 1
}

Write-Host ("WORG_HOME is '{0}'." -f $worgHome)

# ---------------- step 1: ensure Python 3.12 + WORG_PYTHON ----------------
Show-Step "Step 1 - Checking / installing Python 3.12 and WORG_PYTHON"

function Get-Python312Path {
    $pyVersion = & py -3.12 -c "import sys; print(sys.executable)" 2>$null
    if ($LASTEXITCODE -eq 0 -and $pyVersion) { return $pyVersion.Trim() }

    $defaultPath = "$Env:LocalAppData\Programs\Python\Python312\python.exe"
    if (Test-Path -LiteralPath $defaultPath) { return $defaultPath }

    return $null
}

$python312Path = Get-Python312Path

if (-not $python312Path) {
    Write-Host "Python 3.12 not detected." -ForegroundColor Yellow
    Write-Host "Attempt install via winget (usually no admin for per-user installs, but depends on policy)." -ForegroundColor Yellow
    $choice = Read-Host "Install Python 3.12 using winget now? (Y/N)"
    if ($choice -match '^[Yy]') {
        Show-Step "Installing Python 3.12"
        winget install Python.Python.3.12 --source winget --silent
        $python312Path = Get-Python312Path
        if (-not $python312Path) {
            Write-Host "Python 3.12 still not found. Install from python.org, then re-run." -ForegroundColor Red
            exit 1
        }
    } else {
        Write-Host "Please install Python 3.12 (64-bit) from https://www.python.org/downloads/ and re-run." -ForegroundColor Red
        exit 1
    }
}

Write-Host ("Using Python 3.12 at: {0}" -f $python312Path)

Show-Step "Setting WORG_PYTHON environment variable (user-scoped)"
setx WORG_PYTHON "$python312Path" > $null
$Env:WORG_PYTHON = "$python312Path"

Show-Step "Ensuring pip is installed and up to date"
& "$python312Path" -m ensurepip --upgrade
& "$python312Path" -m pip install --upgrade pip

# ---------------- step 2: install Ollama and llama3.2:3b ----------------
Show-Step "Step 2 - Installing Ollama and llama3.2:3b"

function Get-OllamaVersion {
    try {
        $v = & ollama --version 2>$null
        if ($LASTEXITCODE -eq 0 -and $v) { return $v.Trim() }
    } catch { return $null }
    return $null
}

$ollamaVersion = Get-OllamaVersion
if (-not $ollamaVersion) {
    Write-Host "Ollama not detected on PATH." -ForegroundColor Yellow
    $downloadPage = "https://ollama.com/download/windows"
    Write-Host "Opening: $downloadPage"
    Start-Process $downloadPage
    Pause-For-User "Install Ollama, then re-run install-worg.bat (close/reopen terminals so PATH updates apply)."
    exit 1
}

Write-Host ("Ollama detected: {0}" -f $ollamaVersion)
Write-Host "Pulling model llama3.2:3b (may download several GB)..."
& ollama pull llama3.2:3b

# ---------------- step 3: create virtual environment ----------------
Show-Step "Step 3 - Creating virtual environment in WORG_HOME"

$venvPath = Join-Path $worgHome ".venv"
$venvPython = Join-Path $venvPath "Scripts\python.exe"

if (-not (Test-Path -LiteralPath $venvPython)) {
    & "$python312Path" -m venv "$venvPath"
    if ($LASTEXITCODE -ne 0) {
        Write-Host "Failed to create virtual environment in $venvPath" -ForegroundColor Red
        exit 1
    }
}

# ---------------- step 4: install dependencies (no activation) ----------------
Show-Step "Step 4 - Installing Python dependencies from requirements.txt"

$requirementsPath = Join-Path $worgHome "requirements.txt"
if (-not (Test-Path -LiteralPath $requirementsPath)) {
    Write-Host "requirements.txt not found; skipping dependency installation." -ForegroundColor Yellow
} else {
    & "$venvPython" -m pip install --upgrade pip
    & "$venvPython" -m pip install -r "$requirementsPath"
}

Write-Host ""
Write-Host "Done." -ForegroundColor Green
Write-Host "Run later without PowerShell:"
Write-Host ("  {0} {1}" -f $venvPython, (Join-Path $worgHome "worg_proc.py"))
